/*============================================================================
  File:     01_CapturingPlans.sql

  SQL Server Versions: 2008, 2008R2, 2012, 2014, 2016, 2017, 2019
------------------------------------------------------------------------------
  Written by Erin Stellato, SQLskills.com
  
  (c) 2019, SQLskills.com. All rights reserved.

  For more scripts and sample code, check out 
    http://www.SQLskills.com

  You may alter this code for your own *non-commercial* purposes. You may
  republish altered code as long as you include this copyright and give due
  credit, but you must obtain prior permission before blogging this code.
  
  THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF 
  ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED 
  TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.
============================================================================*/

/*
	Enable query store 

*/
ALTER DATABASE [WideWorldImporters] SET QUERY_STORE = ON;
GO
ALTER DATABASE [WideWorldImporters] SET QUERY_STORE (
	OPERATION_MODE = READ_WRITE, INTERVAL_LENGTH_MINUTES = 10
	);
GO

/*
	Optional: Clear out anything that may be in there
	ALTER DATABASE [WideWorldImporters] SET QUERY_STORE CLEAR;
	GO
*/


USE [WideWorldImporters];
GO

/*
	Estimated Plan
	object access
	flow of data (arrows)
	estimates
	missing index notification
*/
SELECT [CustomerID], SUM([AmountExcludingTax])
FROM [Sales].[CustomerTransactions]
WHERE [CustomerID] = 401
GROUP BY [CustomerID];
GO


/*
	re-run to see output
*/
SELECT [CustomerID], SUM([AmountExcludingTax])
FROM [Sales].[CustomerTransactions]
WHERE [CustomerID] = 401
GROUP BY [CustomerID];
GO

/*
	run again with actual plan
	different objects
	estimates and actuals
*/
SELECT [CustomerID], SUM([AmountExcludingTax])
FROM [Sales].[CustomerTransactions]
WHERE [CustomerID] = 401
GROUP BY [CustomerID];
GO

SELECT [CustomerID], SUM([AmountExcludingTax])
FROM [Sales].[CustomerTransactions]
WHERE [CustomerID] = 1092
GROUP BY [CustomerID];
GO



/*
	Create SP to use for testing
*/
USE [WideWorldImporters];
GO

DROP PROCEDURE IF EXISTS [Sales].[usp_CustomerTransactionInfo];
GO

CREATE PROCEDURE [Sales].[usp_CustomerTransactionInfo]
	@CustomerID INT
AS	

	SELECT [CustomerID], SUM([AmountExcludingTax])
	FROM [Sales].[CustomerTransactions]
	WHERE [CustomerID] = @CustomerID
	GROUP BY [CustomerID];
GO

/*
	Create XE session again to capture 
	sql_statement_completed	and sp_statement_completed
	AND query_post_execution_showplan (use with caution!)
*/
IF EXISTS (
	SELECT * 
	FROM sys.server_event_sessions
	WHERE [name] = 'QueryPerf')
BEGIN
	DROP EVENT SESSION [QueryPerf] ON SERVER;
END
GO

CREATE EVENT SESSION [QueryPerf] 
	ON SERVER 
ADD EVENT sqlserver.sp_statement_completed(
	WHERE ([duration]>(1000))),
ADD EVENT sqlserver.sql_statement_completed(
	WHERE ([duration]>(1000))),
ADD EVENT sqlserver.query_post_execution_showplan
ADD TARGET package0.event_file(
	SET filename=N'C:\Temp\QueryPerf',max_file_size=(256))
WITH (
	MAX_MEMORY=16384 KB,EVENT_RETENTION_MODE=ALLOW_SINGLE_EVENT_LOSS,
	MAX_DISPATCH_LATENCY=5 SECONDS,MAX_EVENT_SIZE=0 KB,
	MEMORY_PARTITION_MODE=NONE,TRACK_CAUSALITY=ON,STARTUP_STATE=OFF);
GO

/*
	Check that Query Store is enabled
*/
SELECT 
	[actual_state_desc], 
	[readonly_reason], 
	[desired_state_desc], 
	[current_storage_size_mb], 
    [max_storage_size_mb], 
	[flush_interval_seconds], 
	[interval_length_minutes], 
    [stale_query_threshold_days], 
	[size_based_cleanup_mode_desc], 
    [query_capture_mode_desc], 
	[max_plans_per_query]
FROM [sys].[database_query_store_options];
GO

/*
	Remove everything from the plan cache
*/
DBCC FREEPROCCACHE;
GO

/*
	Start session
*/
ALTER EVENT SESSION [QueryPerf] 
	ON SERVER
	STATE = START;
GO

/*
	Copy below to a separate window 
	We want to keep SSMS query output separate 
	from our demo code
*/

/*
	Enable stats
*/
SET STATISTICS IO ON;
GO
SET STATISTICS TIME ON;
GO
SET STATISTICS XML ON;
GO

/*
	Run SP one time
*/
USE [WideWorldImporters];
GO

EXECUTE [Sales].[usp_CustomerTransactionInfo] 1092;
GO


/*
	Stop event session
*/
ALTER EVENT SESSION [QueryPerf] 
	ON SERVER
	STATE = STOP;
GO


/*
	look at data in 
	XE, cache, and Query Store
	(SSMS data already in another window)
*/


/*
	Query for cache
*/
SELECT
	[qs].[last_execution_time],
	[qs].[execution_count],
	[qs].[total_elapsed_time],
	[qs].[total_elapsed_time]/[qs].[execution_count] [AvgDuration],
	[qs].[total_logical_reads],
	[qs].[total_logical_reads]/[qs].[execution_count] [AvgLogicalReads],
	[t].[text],
	[p].[query_plan]
FROM sys.dm_exec_query_stats [qs]
CROSS APPLY sys.dm_exec_sql_text([qs].sql_handle) [t]
CROSS APPLY sys.dm_exec_query_plan([qs].[plan_handle]) [p]
WHERE [t].[text] LIKE '%usp_CustomerTransactionInfo%';
GO


/*
	Query for Query Store
*/
USE [WideWorldImporters];
GO

SELECT 
	[qsq].[query_id],  
	[qst].[query_sql_text], 
	CASE
		WHEN [qsq].[object_id] = 0 THEN N'Ad-hoc'
		ELSE OBJECT_NAME([qsq].[object_id]) 
	END AS [ObjectName],
	[qsp].[plan_id], 
	[rs].[count_executions],
	[rs].[avg_logical_io_reads], 
	[rs].[avg_duration],
	TRY_CONVERT(XML, [qsp].[query_plan]),
	[rs].[last_execution_time],
	(DATEADD(MINUTE, -(DATEDIFF(MINUTE, GETDATE(), GETUTCDATE())), 
	[rs].[last_execution_time])) AS [LocalLastExecutionTime]
FROM [sys].[query_store_query] [qsq] 
JOIN [sys].[query_store_query_text] [qst]
	ON [qsq].[query_text_id] = [qst].[query_text_id]
JOIN [sys].[query_store_plan] [qsp] 
	ON [qsq].[query_id] = [qsp].[query_id]
JOIN [sys].[query_store_runtime_stats] [rs] 
	ON [qsp].[plan_id] = [rs].[plan_id] 
WHERE OBJECT_NAME([qsq].[object_id]) = 'usp_CustomerTransactionInfo'
GO

/*
	Enable live query statistics and run
*/
EXEC [Sales].[usp_CustomerTransactionInfo] 401;
GO


/*
	Get last executed plan
	(note: this clears the plan cache)
*/
ALTER DATABASE SCOPED CONFIGURATION SET LAST_QUERY_PLAN_STATS = ON;
GO

/*
	re-run the SP
*/
EXEC [Sales].[usp_CustomerTransactionInfo] 1092;
GO


/*
	check sys.dm_exec_query_plan_stats
*/
SELECT
	[qs].[last_execution_time],
	[qs].[execution_count],
	[qs].[total_elapsed_time],
	[qs].[total_elapsed_time]/[qs].[execution_count] [AvgDuration],
	[qs].[total_logical_reads],
	[qs].[total_logical_reads]/[qs].[execution_count] [AvgLogicalReads],
	[t].[text],
	[ps].[query_plan]
FROM sys.dm_exec_query_stats [qs]
CROSS APPLY sys.dm_exec_sql_text([qs].sql_handle) [t]
CROSS APPLY sys.dm_exec_query_plan_stats([qs].[plan_handle]) [ps]
WHERE [t].[text] LIKE '%usp_CustomerTransactionInfo%';
GO


/*
	run the SP one more time...
*/
EXEC [Sales].[usp_CustomerTransactionInfo] 401;
GO


/*
	check sys.dm_exec_query_plan_stats
*/
SELECT
	[qs].[last_execution_time],
	[qs].[execution_count],
	[qs].[total_elapsed_time],
	[qs].[total_elapsed_time]/[qs].[execution_count] [AvgDuration],
	[qs].[total_logical_reads],
	[qs].[total_logical_reads]/[qs].[execution_count] [AvgLogicalReads],
	[t].[text],
	[ps].[query_plan]
FROM sys.dm_exec_query_stats [qs]
CROSS APPLY sys.dm_exec_sql_text([qs].sql_handle) [t]
CROSS APPLY sys.dm_exec_query_plan_stats([qs].[plan_handle]) [ps]
WHERE [t].[text] LIKE '%usp_CustomerTransactionInfo%';
GO


/*
	How to address this isue?
*/

USE [WideWorldImporters]
GO
CREATE NONCLUSTERED INDEX NCI_CustomerTransactions_CustID_AmtExTx
	ON [Sales].[CustomerTransactions] ([CustomerID])
	INCLUDE ([AmountExcludingTax]);
GO

/*
	recompile the SP
*/
sp_recompile [Sales].[usp_CustomerTransactionInfo];
GO

/*
	run the SP one more time...
*/
EXEC [Sales].[usp_CustomerTransactionInfo] 401;
GO

/*
	Drop index
*/
DROP INDEX NCI_CustomerTransactions_CustID_AmtExTx
	ON [Sales].[CustomerTransactions];
GO